<?php
require_once '../config.php';

$input = file_get_contents('php://input');
$data = json_decode($input, true);

$steps = [ 'step1', 'step2', 'save_design', 'set_language' ];

$vars = [
    'set_language' => ['language'],
    'step1'        => ['sole_id'],
    'step2'        => ['generation_type'],
    'design'       => ['generation'],
];

header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');

// Main processing logic
if (!empty($data)) {
    // First, handle authorization
    if (handleAuthorization($data)) {
        checkVariables($data);
    }
} else {
    echo json_encode([
        'error' => 'No data provided'
    ]);
}

// Enhanced error handling and token verification
function handleAuthorization($data) {
    // Check if authorization is required for the current action
    $authRequiredActions = ['step1', 'step2', 'design'];

    // If an action requires authorization, verify the token
    foreach ($authRequiredActions as $action) {
        if (isset($data[$action])) {

            try {

                TokenHelper::verifyBearerToken();
                return true;

            } catch (Exception $e) {
                http_response_code(401);
                echo json_encode([
                    'error' => 'Token verification failed',
                    'details' => $e->getMessage()
                ]);
                exit;
            }
        }
    }

    return true;
}

function checkVariables($data) {
    global $vars;

    // Iterate through possible actions
    foreach ($vars as $action => $variables) {
        if (isset($data[$action])) {
            // Check if all required variables are present
            $hasAllVariables = true;
            foreach ($variables as $variable) {
                if (!isset($data[$variable])) {
                    $hasAllVariables = false;
                    break;
                }
            }

            // If all variables are present, perform the action
            if ($hasAllVariables) {
                switch ($action) {
                    case 'step1':
                        step1($data);
                        break;
                    case 'step2':
                        step2($data);
                        break;
                    case 'save_design':
                        save_design($data);
                        break;
                    case 'set_language':
                        set_language($data);
                }
            } else {
                echo json_encode([
                    'error' => "Not all required variables for action $action are defined"
                ]);
            }
        }
    }
}

// Existing action functions remain mostly the same
function step1($data) {
    SessionHelper::set('sole_id', $data['sole_id']);
    echo json_encode([
        "success" => true,
        "message" => "Step 1: SoleID - " . $data['sole_id']
    ]);
}

function step2($data) {
    // Validate generation type
    $validTypes = ['uppers', 'soles', 'variants'];
    if (empty($data['generation_type']) || !in_array($data['generation_type'], $validTypes)) {
        echo json_encode([
            "success" => false,
            "message" => "Invalid generation type: " . ($data['generation_type'] ?? 'null')
        ]);
        return;
    }
    SessionHelper::set('generation_type', $data['generation_type']);
    echo json_encode([
        "success" => true,
        "message" => "Step 2: GenType - " . $data['generation_type']
    ]);
}

function save_design(array $data) {

    // Initialize the session generations array if it doesn't exist
    if (!SessionHelper::has('saved_generations')) {
        SessionHelper::set('saved_generations', [
            'uppers' => [],
            'soles' => [],
            'variants' => []
        ]);
    }

    if (!isset($data['generation'])) {
        echo json_encode([
            "success" => false,
            "message" => "No generation data provided",
            "receivedData" => $data
        ]);
        return;
    }


    // Extract generation type and image ID from the input data
    $type = $data['generation']['type'] ?? null;
    $imageId = $data['generation']['imageId'] ?? null;

    // Validate the type
    $validTypes = ['uppers', 'soles', 'variants'];
    if (!$type || !in_array($type, $validTypes)) {
        echo json_encode([
            "success" => false,
            "message" => "Invalid generation type",
            "receivedType" => $type
        ]);
        return;
    }

    $savedGenerations = SessionHelper::get('saved_generations');

    if ($imageId && !in_array($imageId, $savedGenerations[$type])) {
        $savedGenerations[$type][] = $imageId;
        SessionHelper::set('saved_generations', $savedGenerations);
    }

    echo json_encode([
        "success" => true,
        "message" => "Generation saved",
        "savedGenerations" => $savedGenerations,
        "receivedData" => $data
    ]);
}

function set_language($data) {
    // Define valid languages
    $USER_LANGUAGES = ['EN', 'RU', 'CN'];

    // Check if language is provided in the request data
    $language = $data['language'] ?? null;

    // Validate language
    if (!$language || !in_array($language, $USER_LANGUAGES)) {
        echo json_encode([
            "success" => false,
            "message" => "Invalid language",
            "receivedLanguage" => $language
        ]);
        return;
    }

    // Save language to session
    SessionHelper::set('language', $language);

    // Respond with success
    echo json_encode([
        "success" => true,
        "message" => "Language updated successfully",
        "language" => $language
    ]);
}