<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/config.php';

// Include the GPUClient service
require_once __DIR__.'/services/GPUClientV002.php';

use Services\GPUClient;

define('CREATIVE_LEVEL', 3);

define('GENERATOR_DATA_DIR', PROJECT_DIR. '/data/images/');

$input = file_get_contents('php://input');
$data = json_decode($input, true);

// Set JSON header for API response
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
$allowedOrigins = ['http://games.local', 'https://markelangelo.com'];

// Validate and set CORS header safely
if (in_array($origin, $allowedOrigins)) {
    header("Access-Control-Allow-Origin: $origin");
} else {
    // Fallback to a default or no origin
    header("Access-Control-Allow-Origin: *");
}

header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');

SessionHelper::startSessionIfNotStarted();

// Rate limiting configuration:
// Define the minimum allowed time interval (in seconds) between consecutive requests
$rateLimitTimeout = 1; // 1/60 second timeout

// Get the current timestamp with microsecond precision
$currentTime = microtime(true);

// Check if there's a record of a previous request
if (SessionHelper::has('last_request_time') && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $elapsed = $currentTime - SessionHelper::get('last_request_time');
    if ($elapsed < $rateLimitTimeout) {
        // If requests are too frequent, block this request with an error response
        http_response_code(429); // HTTP status 429: Too Many Requests
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => 'Too many requests. Please wait a moment before retrying.'
        ]);
        exit;
    }
}

function handleAuthorization($data) {
    // Check if authorization is required for the current action
    $authRequiredActions = ['run',];

    // If an action requires authorization, verify the token
    foreach ($authRequiredActions as $action)
    {
        if (isset($data[$action])) {
            TokenHelper::verifyBearerToken();
        }
    }

    return true;
}

if (handleAuthorization($data)) {

    $user_id = SessionHelper::get('FURBA_id');

    if (!$user_id) {
        // User not authenticated
        http_response_code(401);
        echo json_encode([
            'success' => false,
            'message' => 'Not authenticated'
        ]);
        exit;
    }

    $sole_id = SessionHelper::get('sole_id');

    if (!$sole_id) {
        // User not authenticated
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'Parameter not found'
        ]);
        exit;
    }

    $generation_type = SessionHelper::get('generation_type');

    // Find the sales leader matching the current sole_id
    $selectedSalesLeader = array_filter(SALES_LEADERS, function($salesLeader) use ($sole_id) {
        return $salesLeader['id'] == $sole_id;
    });

    $selectedSalesLeader = array_map(fn($item) => $item, $selectedSalesLeader);
    $selectedSalesLeader = reset($selectedSalesLeader);

    if (count($selectedSalesLeader) && $selectedSalesLeader['id'] === $sole_id) {

        $description_fn = GENERATOR_DATA_DIR . $selectedSalesLeader['generation']['description'];
        $prompts = getPrompt($description_fn);

        $initial = $selectedSalesLeader['generation']['initial'];

        $prompt = $selectedSalesLeader['generation']['shoe_type'] .' '.
            (CREATIVE_LEVEL > 1? getRandomColorScheme() .' ': '').
            (CREATIVE_LEVEL > 2? getRandomDetails() .' ': '');

        $mask = '';
        switch ($generation_type)
        {

            case 'soles':
                $mask = $selectedSalesLeader['generation']['mask_sole'];
                $prompt .= ($prompts['description']?? ''). $prompts['sole']?? '';
                break;

            case 'uppers':
                $mask = $selectedSalesLeader['generation']['mask_upper'];
                $prompt .= ($prompts['description']?? ''). $prompts['upper']?? '';
                break;

            case 'variants':
            default:
                $mask = $selectedSalesLeader['generation']['mask_variants']?? 'variants_transparent.png';
                $prompt .= $prompts['variants']?? '';
                break;
        }

        $job_data = ['i' => $initial, 'm' => $mask, 'p' => $prompt];

        $jobId = setJob($job_data);

        if ($jobId) {
            SessionHelper::set('jobId', $jobId);
        }

        echo json_encode([
            'status' => 'STARTED',
            'jobId' => $jobId
        ]);

    }
    else {
        echo 'Sole not found';
    }
}

function setJob($job_data) {

    try {
        $seed = rand(999, 1000000);

        $_initialImageName = $job_data['i'];
        $_maskImageName = $job_data['m'];
        $_positive_prompt = trim($job_data['p']);

        if (empty($_positive_prompt)) {
            throw new Exception('Prompt must be properly setup');
        }

        $prompts = [
            'POSITIVE_PROMPT' => $_positive_prompt. ' ' . DEFAULT_POSITIVE_PROMPT, //' 8k, 16k, photorealistic, high quality, detailed'
            'NEGATIVE_PROMPT' => 'jpeg artifacts, text, logo, label, trademark, signature, '.
            'duplicate, tiling, monochrome, bad art, 3d, CG, sketch, painting, cartoon, art, '.
            'canvas, drawing, cropped, watermark, username, collage, artist_name, signature, '.
            'artist_logo, deformed, disfigured, poorly drawn, blurry, ugly, (low quality), '.
            '(worst quality), (poor quality), (bad quality), low resolution, pixelated, faded, '.
            'distorted, blurry, blurred, grainy, grainy, noisy, (washed out), overexposed, '.
            'underexposed, (out of focus), glitchy, scratched, tarnished, smudged, cracked, '.
            'warped, overcompressed, misaligned, oily, shiny, double shoe, '.
            'pair of shoes, left-footed shoe'
        ];

        // Define paths relative to this file
        $images_path = GENERATOR_DATA_DIR;
        // Verify paths exist
        if (!is_dir($images_path)) {
            throw new Exception('Images directory not found: ' . $images_path);
        }

        $initImagePath = $images_path . $_initialImageName;
        $maskImagePath = $images_path . $_maskImageName;

        // Verify files exist
        if (!file_exists($initImagePath)) {
            throw new Exception('Initial image not found: ' . $initImagePath);
        }
        if (!file_exists($maskImagePath)) {
            throw new Exception('Mask image not found: ' . $maskImagePath);
        }

        // Read and encode images
        $initImageBase64 = base64_encode(file_get_contents($initImagePath));
        $maskImageBase64 = base64_encode(file_get_contents($maskImagePath));

        $images = [
            [
                'name' => 'initial_image.png',
                'image' => $initImageBase64
            ],
            [
                'name' => 'mask_image.png',
                'image' => $maskImageBase64
            ]
        ];

        $request = GPUClient::createRequest($seed, $prompts, $images);
        $response = GPUClient::sendRequest($request);

        //Example Check status periodically
        //$status = GPUClient::checkStatus($response['id']);

        if (!isset($response['id'])) {
            throw new Exception('No job ID in response');
        }

        // Store job information
        $jobId = $response['id'];
        $tempFile = sys_get_temp_dir() . '/gpu_job_' . $jobId . '.txt';

        $jobData = [
            'jobId' => $jobId,
            'attempts' => 0,
            'lastCheck' => time(),
            'status' => 'PENDING',
            'created' => date('Y-m-d H:i:s'),
            'i' => $_initialImageName,
            'm' => $_maskImageName,
            'p' => $prompts['POSITIVE_PROMPT'],
        ];

        if (!file_put_contents($tempFile, json_encode($jobData))) {
            throw new Exception('Failed to write temporary file');
        }

        return $jobId;

    } catch (Exception $e) {
        // Log error for debugging
        error_log('GPU Job Error: ' . $e->getMessage());

        // Return error response
        http_response_code(500);
        echo json_encode([
            'status' => 'ERROR',
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

function getPrompt(string $description_filename) {
    $description = (file_exists($description_filename) && is_readable($description_filename))? file_get_contents($description_filename): '';
    if (empty($description)) {
        return;
    }

    $description_sections = ['#Description', '#Upper', '#Sole', '#Info', '#Variants'];
    $sections = [];

    // Split the description text into sections based on predefined section keys
    foreach ($description_sections as $index => $section) {
        // Find the start position of the current section
        $start = strpos($description, $section);

        // Find the start position of the next section (or end of text)
        $end = ($index < count($description_sections) - 1)
            ? strpos($description, $description_sections[$index + 1], $start)
            : strlen($description);

        if ($start !== false) {
            // Extract the section content
            $content = trim(substr(
                $description,
                $start + strlen($section),
                ($end !== false ? $end - $start - strlen($section) : null)
            ));

            // Remove the section key from the content and trim
            $section_key = strtolower(str_replace('#', '', $section));
            $sections[$section_key] = trim($content);
        }
    }

    // Return the sections array
    return $sections;
}

function getRandomColorScheme(): string {
    $_colors = [
        'Red', 'Green', 'Blue', 'Yellow', 'Orange', 'White', 'Black',
        'Purple', 'Pink', 'Brown', 'Gray', 'Teal', 'Magenta',
        'Lavender', 'Turquoise', 'Maroon', 'Navy', 'Olive', 'Cyan',
        'Indigo', 'Coral', 'Salmon', 'Mint', 'Periwinkle', 'Crimson',
        'Gold', 'Silver', 'Bronze', 'Emerald', 'Sapphire', 'Ruby',
        'Chartreuse', 'Burgundy', 'Aquamarine', 'Slate', 'Beige',
        'Mustard', 'Forest Green', 'Sky Blue', 'Rose', 'Plum'
    ];

    $_palletes = [
        'calm',
        'rich',
        'vivid',
        'stylish',
        'elegant',
        'sophisticated',
        'minimalist',
        'bold',
        'serene',
        'vibrant',
        'muted',
        'dramatic'
    ];

    // Randomly select a color and a palette to create a unique color scheme description
    $randomColor = $_colors[array_rand($_colors)];
    $randomPalette = $_palletes[array_rand($_palletes)];

    // Return a formatted color scheme description
    return "in {$randomPalette} {$randomColor} color scheme";
}

function randomElements($num = 1) {
    // Define elements array
    $elements = [
        'Laces', 'Zipper', 'Buckle', 'Velcro Strap', 'Elastic Band',
        'Decorative Rivets', 'Embroidered Patch', 'Metal Eyelets',
        'Leather Trim', 'Reflective Accents', 'Perforated Panels',
        'Contrast Stitching'
    ];

    // Validate input
    $num = max(1, min($num, count($elements)));

    // Use array_rand for efficient random selection
    $randomKeys = array_rand($elements, $num);

    // Ensure $randomKeys is always an array
    $randomKeys = is_array($randomKeys) ? $randomKeys : [$randomKeys];

    // Select random elements
    $randomElements = array_map(function($key) use ($elements) {
        return $elements[$key];
    }, $randomKeys);

    // Return as comma-separated string or single element
    return count($randomElements) > 1
        ? implode(', ', $randomElements)
        : $randomElements[0];
}

function getRandomDetails() {
    $_details = [
        'Breathable Mesh Upper',
        'Shock-Absorbing Midsole',
        'Ergonomic Arch Support',
        'Lightweight Performance Fabric',
        'Cushioned Heel Counter',
        'Flexible Outsole',
        'Waterproof Membrane',
        'Anti-Slip Tread Pattern',
        'Memory Foam Insole',
        'Anatomical Toe Box',
        'Reinforced Heel Cup',
        'Quick-Dry Lining',
        'Impact-Resistant Toe Cap',
        'Ankle Stabilization',
        'Moisture-Wicking Material',
        'Orthopedic Heel Cushioning',
        'Seamless Construction',
        'Compression Fit Design',
        'Thermal Insulation',
        'Vibration-Dampening Technology',
        'Anatomical Heel Contour',
        'Ventilation Channels',
        'Adaptive Fit System',
        'Energy Return Foam',
        'Lateral Support Panels',
        'Precision Fit Lacing',
        'Multi-Directional Grip',
        'Lightweight Composite Shank',
        'Temperature-Regulating Fabric',
        'Dynamic Flex Zones',
        'Pressure-Mapped Cushioning',
        'Eco-Friendly Recycled Materials',
        'Biomechanical Design',
        'Zero-Drop Sole Construction',
        'Responsive Cushioning',
        'Durability-Enhanced Overlays',
        'Terrain-Specific Traction',
        'Minimalist Structural Support',
        'Advanced Foot Mapping',
        'Sustainable Performance Materials',
        'Precision-Engineered Sole',
        'Adaptive Climate Control',
        'Ergonomic Movement Design',
        'Shock Distribution Technology',
        'Performance-Driven Construction',
        'Intelligent Material Composition',
        'Biomimetic Sole Design',
        'Advanced Comfort Engineering'
    ];

    $randomDetail = $_details[array_rand($_details)];

    return $randomDetail . ' with '. randomElements(2) .'. ';
}