<?php
namespace Helpers;

/**
 * Класс для добавления мета-информации в PNG-файл.
 */
class PNGMeta {

    /**
     * Добавляет meta-информацию в PNG-файл.
     *
     * Принимает бинарные данные PNG и массив meta-информации.
     * Каждая пара (ключ => значение) будет сохранена как текстовый chunk в PNG.
     *
     * @param string $pngData Бинарные данные PNG.
     * @param array  $metaData Ассоциативный массив meta-документации,
     *                         например: ['Comment' => 'Generated by GPUClient', 'Author' => 'YourName'].
     * @return string Бинарные данные PNG с добавленной мета-информацией.
     * @throws \Exception Если Imagick не установлен or возникла ошибка при обработке.
     */
    public static function addMetaData(string $pngData, array $metaData): string {
        if (!extension_loaded('imagick')) {
            throw new \Exception("Расширение Imagick обязательно для работы с мета-информацией PNG.");
        }

        // Загружаем изображение из бинарного потока.
        $image = new \Imagick();
        $image->readImageBlob($pngData);

        // Перебираем переданный массив meta-информации.
        foreach ($metaData as $key => $value) {
            // Добавляем текстовый chunk. Ключ лучше префиксовать "png:",
            // чтобы Imagick обработал его корректно для PNG.
            $image->setImageProperty('png:' . $key, $value);
        }

        // Получаем итоговые бинарные данные изображения.
        $pngWithMeta = $image->getImagesBlob();
        $image->destroy();

        return $pngWithMeta;
    }
}