<?php
class UserAuthorizer
{
    /**
     * Class UserAuthorizer
     */
    private UserManager $userManager;

    /**
     * Constructor.
     *
     * Ensures that the PHP session is started and initializes UserManager.
     *
     * @param string $databasePath Path to the SQLite database file
     */
    public function __construct(string $databasePath)
    {
        // Start session if it hasn't been started already.
        SessionHelper::startSessionIfNotStarted();

        // Initialize UserManager with the provided database path
        $this->userManager = new UserManager($databasePath);
    }

    /**
     * Check user credentials and set up session if valid
     *
     * @param string $username Submitted username
     * @param string $password Submitted password
     * @return array Authentication result
     */
    public function checkUser(string $username, string $password): array
    {
        // Validate input fields are not empty
        if (empty($username) || empty($password)) {
            return [
                'success' => false,
                'message' => 'Login failed', //Authorization details are required.'
            ];
        }

        // Attempt to authenticate user
        $user = $this->userManager->authenticateUser($username, $password);

        if ($user) {
            return [
                'success' => true,
                'message' => 'Login successful',
                'user' => [
                    'id' => $user['id'],
                    'username' => $user['username']
                ]
            ];
        } else {
            return [
                'success' => false,
                'message' => 'Login failed'
            ];
        }
    }

    /**
     * Check if the user is authorized.
     *
     * This method checks for the existence of 'user_id' in the session.
     *
     * @return bool Returns true if a user is logged in, false otherwise.
     */
    public function isAuthorized(): bool
    {
        return SessionHelper::has('FURBA_id');
    }

    /**
     * Enforce user authorization.
     *
     * If the user is not authorized, this method sends a 401 Unauthorized response
     * with a JSON error message and terminates the script.
     */
    public function enforceAuthorization(): void
    {
        if (!$this->isAuthorized()) {
            http_response_code(403);
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'message' => 'Unauthorized access'
            ]);
            exit;
        }
    }

    /**
     * Handle unauthorized access
     *
     * Redirects to login page if user is not authorized
     *
     * @param string $redirectUrl Optional URL to redirect after login
     */
    public function handleUnauthorizedAccess(string $redirectUrl = ''): void
    {
        // If not authorized, redirect to login page
        if (!$this->isAuthorized()) {
            // Store the original destination for post-login redirect
            if (!empty($redirectUrl)) {
                SessionHelper::set('redirect_after_login', $redirectUrl);
            }

            // Redirect to login page
            header('Location: ' + $redirectUrl?? './');
            exit;
        }
    }
}