<?php
class TokenHelper {
    // Secret key for token generation and verification
    private const SECRET_KEY = 'IJCKLJ#9jdfkljKAJSD3JLDjs82nLDJs92hlhas()(_jksdjs';

    /**
     * Generate a secure authentication token
     *
     * @param int $userId User identifier
     * @param string $role User role (default: 'user')
     * @param int $expireTime Token expiration time (default: 1 hour)
     * @return string Generated token
     */
    public static function createToken(
        int $expireTime = (60 * 60 * 24)
    ): string {

        // Prepare token payload
        $payload = [
            'timestamp' => time(),
            'expire_time' => time() + $expireTime
        ];

        // Generate cryptographically secure signature
        $signature = hash_hmac(
            'sha256',
            json_encode($payload),
            self::SECRET_KEY,
            true
        );

        // Add signature to payload
        $payload['signature'] = base64_encode($signature);
        // Encode and return token
        return base64_encode(json_encode($payload));
    }

    /**
     * Verify and decode authentication token
     *
     * @return array|false Decoded token payload or false if invalid
     */
    public static function verifyToken($token) {
        try {

            if (empty($token)) {
                return false;
            }

            // Decode the token
            $decodedPayload = json_decode(base64_decode($token), true);

            // Validate basic token structure
            if (!$decodedPayload) {
                return false;
            }

            // Check token expiration
            if ($decodedPayload['expire_time'] < time()) {
                return false;
            }

            // Verify signature
            $expectedSignature = hash_hmac(
                'sha256',
                json_encode(array_diff_key( $decodedPayload, ['signature' => ''])),
                self::SECRET_KEY,
                true
            );

            $providedSignature = base64_decode($decodedPayload['signature']);

            // Compare signatures using timing-safe comparison
            if (!hash_equals($expectedSignature, $providedSignature)) {
                return false;
            }

            return true;

        } catch (Exception $e) {
            // Log the error in a production environment
            return false;
        }
    }

    /**
     * Refresh an existing token
     *
     * @param int $newExpireTime New expiration time
     * @return string|false New token or false if refresh fails
     */
    public static function refreshToken($token, int $expireTime = (60 * 60 * 24)) {

        if (!self::verifyToken($token)) {
            // Create new token with same user details but updated expiration
            return self::createToken($expireTime);
        }

        return $token;
    }


    public static function verifyBearerToken() {

        try {
            $token = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
            $token = str_replace('Bearer ', '', $token);

            if (!self::verifyToken($token)) {

                // Attempt to refresh the token
                $newToken = self::refreshToken($token);

                // If refresh fails, return unauthorized
                if ($newToken === false) {
                    http_response_code(401);
                    echo json_encode([
                        'error' => 'Invalid or expired token',
                        'message' => 'Authentication failed'
                    ]);
                    exit;
                }

                // Token successfully refreshed
                // Here you would typically update the token in session and local storage
                // This is a placeholder - actual implementation depends on your frontend framework
                return [
                    'status' => 'refreshed',
                    'at' => $newToken
                ];
            }

            return [
                'status' => 'ok',
                'at' => $token
            ];

        } catch (Exception $e) {
            http_response_code(401);
            echo json_encode([
                'error' => 'Token verification failed',
                'details' => $e->getMessage()
            ]);
            exit;
        }
    }

}

// Example usage in authorization middleware
/*
class AuthMiddleware {
    public function authenticate() {
        $tokenData = TokenHelper::verifyToken();

        if (!$tokenData) {
            // Unauthorized access
            http_response_code(401);
            echo json_encode(['error' => 'Invalid or expired token']);
            exit;
        }

        // Store user context for further processing
        $_SESSION['user'] = $tokenData;
        return true;
    }
}*/