<?php

/**
 * Example usage of SessionHelper methods
 *
 * These examples demonstrate how to use the static methods
 * of the SessionHelper class in different scenarios.

public static function exampleUsage(): void
{
    // Start a session if not already started
    self::startSessionIfNotStarted();

    // Set a user ID in the session
    self::set('user_id', 123);

    // Check if a key exists in the session
    if (self::has('user_id')) {
        // Get the user ID from the session
        $userId = self::get('user_id');
        echo "User ID: $userId\n";
    }

    // Set multiple session variables
    self::set('username', 'johndoe');
    self::set('is_logged_in', true);

    // Get a session variable with a default value
    $email = self::get('email', 'no_email@example.com');

    // Remove a specific session variable
    self::remove('username');

    // Regenerate session ID for security
    self::regenerateId();

    // Completely clear the session
    // self::clear();
}
*/

class SessionHelper
{
    public static function startSessionIfNotStarted(): void
    {
        if (session_status() === PHP_SESSION_NONE) {
            // Enhanced cookie settings
            session_set_cookie_params([
                'lifetime' => 604800,  // 7 days
            ]);
            session_start();
        }
    }

    public static function getSessionId(): string
    {
        self::startSessionIfNotStarted();
        return session_id();
    }

    public static function set(string $key, $value): void
    {
        self::startSessionIfNotStarted();
        $_SESSION[$key] = $value;
    }

    public static function get(string $key, $default = null)
    {
        self::startSessionIfNotStarted();
        return $_SESSION[$key] ?? $default;
    }

    public static function has(string $key): bool
    {
        self::startSessionIfNotStarted();
        return isset($_SESSION[$key]);
    }

    public static function remove(string $key): void
    {
        self::startSessionIfNotStarted();
        unset($_SESSION[$key]);
    }

    public static function clear(): void
    {
        self::startSessionIfNotStarted();
        $_SESSION = [];
    }

    public static function regenerateId(): void
    {
        self::startSessionIfNotStarted();
        session_regenerate_id(true);
    }

    public static function updateSession(): bool {
        self::startSessionIfNotStarted();

        $currentTime = time();
        $sessionMaxLifetime = 60 * 30;
        // Check session age
        $sessionCreated = self::get('created', 0);

        // Periodically regenerate session ID
        if ($currentTime - $sessionCreated > $sessionMaxLifetime) { // Every 30 minutes
            self::set('created', $currentTime);
            self::regenerateCsrfToken();
            return true;
        }

        return false;
    }

    public static function regenerateCsrfToken(): string {
        self::startSessionIfNotStarted();

        $sessionMaxLifetime = 86400; // 24 hours

        $csrfToken = bin2hex(random_bytes(32));
        $tokenExpiry = time() + $sessionMaxLifetime; // 1-hour token validity

        self::set("csrfToken", $csrfToken);
        self::set("csrfTokenExpiry", $tokenExpiry);

        return $csrfToken;
    }

    public static function validateCsrfToken(string $token): bool {
        $storedToken = self::get('csrfToken');
        $tokenExpiry = self::get('csrfTokenExpiry', 0);

        return (
            hash_equals($storedToken, $token) &&
            time() <= $tokenExpiry
        );
    }


    public static function generateFingerprint(): string {
        $fingerprint = hash('sha256',
            $_SERVER['HTTP_USER_AGENT'] .
            $_SERVER['REMOTE_ADDR']
        );
        self::set('session_fingerprint', $fingerprint);
        return $fingerprint;
    }

    public static function validateFingerprint(): bool {
        $currentFingerprint = self::generateFingerprint();
        $storedFingerprint = self::get('session_fingerprint');

        return $currentFingerprint === $storedFingerprint;
    }

    public static function destroy(): void {
        self::startSessionIfNotStarted();

        // Clear all session data
        $_SESSION = [];

        // Destroy server-side session
        session_unset();
        session_destroy();
    }
}