<?php
class ProfileManager
{
    private ?PDO $pdo;

    /**
     * Constructor
     *
     * @param string $databasePath Path to the SQLite database file
     */
    public function __construct(string $databasePath)
    {
        SessionHelper::startSessionIfNotStarted();
        if (!file_exists($databasePath)) {
            die("DB not found");
        }
        $this->pdo = new PDO("sqlite:$databasePath");
        $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }

    /**
     * Get all profiles from the database
     *
     * @return array All profile data
     */
    public function getAllProfiles(): array
    {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM profiles");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            echo "Error getting all profiles: " . $e->getMessage() . "\n";
            return [];
        }
    }

    /**
     * Create a new profile in the database
     * Role ID by default equal 3 - Demo user
     * @param int $userId User ID
     * @param array $data Profile data
     * @return int Profile ID
     */
    public function createProfile(int $userId, array $data): int
    {
        try {
            $firstName = $data['first_name'] ?? null;
            $lastName = $data['last_name'] ?? null;

            if (!$userId || is_null($firstName) || is_null($lastName)) {
                throw new Exception("Invalid user ID or profile data");
            }

            $stmt = $this->pdo->prepare("INSERT INTO profiles (user_id, first_name, last_name) VALUES (:user_id, :first_name, :last_name)");
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':first_name', $firstName);
            $stmt->bindParam(':last_name', $lastName);
            $stmt->execute();
            return $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            echo "Error creating profile: " . $e->getMessage() . "\n";
            return 0;
        } catch (Exception $e) {
            echo "Error creating profile: " . $e->getMessage() . "\n";
            return 0;
        }
    }

    /**
     * Update a profile in the database
     *
     * @param int $id Profile ID
     * @param array $data Profile data to update
     * @return bool True if the update was successful, false otherwise
     */
    public function updateProfile(int $id, array $data): bool
    {
        try {

            $userId = $data['user_id'] ?? null;
            $firstName = $data['first_name'] ?? null;
            $lastName = $data['last_name'] ?? null;
            $roleId = $data['role_id'] ?? null;

            $stmt = $this->pdo->prepare("UPDATE profiles SET user_id = :user_id, first_name = :first_name, last_name = :last_name, role_id = :role_id WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':first_name', $firstName);
            $stmt->bindParam(':last_name', $lastName);
            $stmt->bindParam(':role_id', $roleId);
            $stmt->execute();
            return true;
        } catch (PDOException $e) {
            echo "Error updating profile: " . $e->getMessage() . "\n";
            return false;
        }
    }

    /**
     * Delete a profile from the database
     *
     * @param int $id Profile ID
     * @return bool True if the deletion was successful, false otherwise
     */
    public function deleteProfile(int $id): bool
    {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM profiles WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            return true;
        } catch (PDOException $e) {
            echo "Error deleting profile: " . $e->getMessage() . "\n";
            return false;
        }
    }

    /**
     * Get a profile from the database
     *
     * @param int $id Profile ID
     * @return array Profile data
     */
    public function getProfile(int $id): array
    {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM profiles WHERE user_id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            echo "Error getting profile: " . $e->getMessage() . "\n";
            return [];
        }
    }

    /**
     * Close the database connection
     */
    public function __destruct()
    {
        $this->pdo = null;
    }
}