<?php
require_once './config.php';
require_once './services/GPUClientV002.php';
require_once 'saveGeneratedImage.php';

/**
 * GPU Job Status Checker
 *
 * This script implements a long-polling mechanism for checking GPU job status.
 * It uses temporary files to maintain state between requests and implements
 * rate limiting to prevent overwhelming the GPU service.
 */

use Services\GPUClient;

define('DEFAULT_TIMEOUT_SECONDS', 10);

if (!IS_LOCAL) {
    require_once './helpers/cors.php';
}
header('Content-Type: application/json');

// Validate job ID from request parameters
$jobId = $_GET['jobId'] ?? null;
if (!$jobId) {
    die(json_encode(['error' => 'No job ID provided']));
}

// Locate and validate the job's temporary file
$tempFile = sys_get_temp_dir() . '/gpu_job_' . $jobId . '.txt';
if (!file_exists($tempFile)) {
    die(json_encode(['error' => 'Job not found']));
}

// Load current job state from temporary file
$jobData = json_decode(file_get_contents($tempFile), true);

// Implement rate limiting: ensure at least 10 seconds between status checks
// This prevents overwhelming the GPU service with too frequent requests
if ((time() - $jobData['lastCheck'] < DEFAULT_TIMEOUT_SECONDS) && $jobData['attempts'] > 0) {
    echo json_encode([
        'status' => $jobData['status'],
        'message' => 'Rate limited - please wait before next check',
        'nextCheckIn' => DEFAULT_TIMEOUT_SECONDS - (time() - $jobData['lastCheck'])
    ]);
    exit();
}
else {
    sleep(DEFAULT_TIMEOUT_SECONDS);
}

try {
    // Request current status from GPU service
    $statusResponse = GPUClient::checkStatus($jobId);
    $jobData['attempts']++;
    $jobData['lastCheck'] = time();

    if (isset($statusResponse['status'])) {
        $jobData['status'] = $statusResponse['status'];

        // Handle completed jobs
        if ($statusResponse['status'] === 'COMPLETED') {
            $jobData['output'] = $statusResponse['output'] ?? null;
            $jobData['completedAt'] = date('Y-m-d H:i:s');

            // Save the image and get the path
            $savedImagePath = saveGeneratedImage($jobData, $jobId);

            // Add the saved image path to the response if successful
            if ($savedImagePath) {
                $jobData['savedImagePath'] = $savedImagePath;
                // Update the temp file with the image path
            }

            // Keep the temp file for debugging purposes
            file_put_contents($tempFile, json_encode($statusResponse));

            //'output' => $statusResponse['output'] ?? null,
            echo json_encode([
                'status' => 'COMPLETED',
                'jobHistory' => $jobData
            ]);
            exit;
        }
        // Handle failed jobs
        elseif ($statusResponse['status'] === 'FAILED') {
            $jobData['error'] = $statusResponse['error'] ?? 'Unknown error';
            $jobData['failedAt'] = date('Y-m-d H:i:s');

            // Keep the temp file for debugging purposes
            file_put_contents($tempFile, json_encode($jobData));

            echo json_encode([
                'status' => 'FAILED',
                'error' => $statusResponse['error'] ?? 'Unknown error',
                'jobHistory' => $jobData
            ]);
            exit;
        }
    }

    // Update temp file with latest status
    file_put_contents($tempFile, json_encode($jobData));

    // Return current status and attempt information
    echo json_encode([
        'status' => $jobData['status'],
        'attempts' => $jobData['attempts'],
        'lastCheck' => date('Y-m-d H:i:s', $jobData['lastCheck']),
        'created' => $jobData['created']
    ]);

} catch (\Exception $e) {
    // Log the error state to the temp file
    $jobData['lastError'] = $e->getMessage();
    $jobData['lastErrorAt'] = date('Y-m-d H:i:s');
    file_put_contents($tempFile, json_encode($jobData));

    echo json_encode([
        'status' => 'ERROR',
        'message' => $e->getMessage(),
        'jobHistory' => $jobData
    ]);
}
