<?php
use Services\GPUClient;

require_once '../config.php';

// Include the GPUClient service
require_once SERVICES_DIR . '/GPUClientV002.php';
require_once __DIR__.'/save2file.php';

define('DEFAULT_TIMEOUT_SECONDS', 10);

$input = file_get_contents('php://input');
$data = json_decode($input, true);

header('Content-Type: application/json');

try {

    // Validate job ID from request parameters
    $jobId = $_GET['jobId'] ?? null;
    if (!$jobId) {
        throw new Exception('No job ID provided');
    }

    // Locate and validate the job's temporary file
    $tempFile = sys_get_temp_dir() . '/gpu_job_' . $jobId . '.txt';
    if (!file_exists($tempFile)) {
        throw new Exception('Job not found');
    }

    // Load current job state from temporary file
    $jobData = json_decode(file_get_contents($tempFile), true);

    if ($jobData['status'] === 'COMPLETED') {

        // Delete temporary file after job completion
        if (file_exists($tempFile)) {
            unlink($tempFile);
        }

        echo json_encode([
            'status' => $jobData['status']
        ]);

        exit();
    }

    // Implement rate limiting: ensure at least 10 seconds between status checks
    // This prevents overwhelming the GPU service with too frequent requests

    if (!empty($jobData['lastCheck']) && (time() - $jobData['lastCheck'] < DEFAULT_TIMEOUT_SECONDS) && $jobData['attempts'] > 0) {
        echo json_encode([
            'status' => $jobData['status'],
            'message' => 'Rate limited - please wait before next check',
            'nextCheckIn' => DEFAULT_TIMEOUT_SECONDS - (time() - $jobData['lastCheck'])
        ]);
        exit();
    }
    else {
        sleep(DEFAULT_TIMEOUT_SECONDS);
    }
// ========================================================
// Requesting current status from GPU service
// ========================================================
    $statusResponse = GPUClient::checkStatus($jobId);
    $jobData['attempts']++;
    $jobData['lastCheck'] = time();

    if (isset($statusResponse['status'])) {
        $jobData['status'] = $statusResponse['status'];

        // Handle completed jobs
        if ($statusResponse['status'] === 'COMPLETED') {
            $jobData['output'] = $statusResponse['output'] ?? null;
            $jobData['completedAt'] = date('Y-m-d H:i:s');

            // Save the image and get the path
            $savedImagePath = saveGeneratedImage($jobData, $jobId);

            file_put_contents($tempFile, json_encode($jobData));

            unset($jobData['output']);

            // Add the saved image path to the response if successful
            if ($savedImagePath) {
                $jobData['user_id'] = SessionHelper::get('FURBA_id', '1');
                $jobData['savedImagePath'] = $savedImagePath;
                $jobData['image'] = str_replace(PROJECT_DIR . '/data/generations/', '', $savedImagePath);
                // Update the temp file with the image path
            }

            //'output' => $statusResponse['output'] ?? null,
            echo json_encode([
                'status' => 'COMPLETED',
                'jobId' => $jobId,
                'image'=> $jobData['image'],
                //'jobHistory' => $jobData
            ]);
            exit;
        }

        // Handle failed jobs
        elseif ($statusResponse['status'] === 'FAILED') {
            $jobData['error'] = $statusResponse['error'] ?? 'Unknown error';
            $jobData['failedAt'] = date('Y-m-d H:i:s');

            //if (file_exists($tempFile)) {
            //    unlink($tempFile);
            //}
            // Keep the temp file for debugging purposes
            file_put_contents($tempFile, json_encode($jobData));

            echo json_encode([
                'status' => 'FAILED',
                'error' => $statusResponse['error'] ?? 'Unknown error',
                'jobHistory' => $jobData
            ]);
            exit;
        }
    }

    // Update temp file with latest status
    file_put_contents($tempFile, json_encode($jobData));

    // Return current status and attempt information
    echo json_encode([
        'status' => $jobData['status'],
        'attempts' => $jobData['attempts'],
        'lastCheck' => date('Y-m-d H:i:s', $jobData['lastCheck']),
        'created' => $jobData['created']
    ]);

} catch (Exception $e) {
    // Log error for debugging
    error_log('GPU Job Error: ' . $e->getMessage());

    // Return error response
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode([
        'status' => 'ERROR',
        'message' => $e->getMessage()
    ]);
}