<?php
/**
 *
 * This class generates a file name from form input data.
 *
 * It replicates the JavaScript function by:
 *  1. Storing the form data internally and retrieving each value using getValue(), which trims the value
 *     or returns the default if the key is not set.
 *  2. Processing the "season" field. If any season value is "0000", the final season string becomes "0000".
 *     Otherwise, each season string (padded to 4 digits) is processed digit-by-digit and summed, capping each digit at 1.
 *  3. Assembling the final file name in the format:
 *     product_type_image_type_image_source_country_code_product_kind_product_number_market_focus_group_style_production_type_material_season_version_brand.png
 */
class FNGenerator {

    /**
     * Holds the form data for internal use.
     *
     * @var array
     */
    private $data = [];

    /**
     * Retrieves and trims a scalar value from the form data.
     * If the value is not set, returns the default value.
     *
     * @param string $key The key to look up in form data.
     * @param mixed $default The default value if key is not set.
     * @return string The trimmed value or the default.
     */
    private function getValue($key, $default = '') {
        return isset($this->data[$key]) ? trim($this->data[$key]) : $default;
    }

    /**
     * Generates the file name string from provided form data.
     *
     * Expected keys in $data:
     * - product_type, country_code, product_kind, product_number, market,
     *   focus_group, style, production_type, material, version, brand
     * - image_type, image_source (radio buttons)
     * - season (an array of season values)
     *
     * @param array $data Associative array with form input values.
     * @return string The assembled file name with a .png extension.
     */
    public function getName(array $data) {
        // Save form data to class property for use with getValue().
        $this->data = $data;

        // Use getValue() to retrieve and trim each scalar value.
        $productType    = $this->getValue('product_type', 'Unknown_type');
        $countryCode    = $this->getValue('country_code', '00');
        $productKind    = $this->getValue('product_kind', '00');
        $productNumber  = $this->getValue('product_number', '0000');
        $market         = $this->getValue('market', 'Massmarket');
        $focusGroup     = $this->getValue('focus_group', 'Men');
        $style          = $this->getValue('style', '01');
        $productionType = $this->getValue('production_type', '00');
        $material       = $this->getValue('material', '000');
        $version        = $this->getValue('version', '000001');
        $brand          = $this->getValue('brand', 'noname');

        // Retrieve radio button values.
        $imageType   = $this->getValue('image_type', 'initial'); //initial, mask, plug, description
        $imageSource = $this->getValue('image_source', 'rl'); //rl, ai, sk = sketch

        // Retrieve season values; expected as an array.
        $seasons = (isset($this->data['season']) && is_array($this->data['season']))
            ? array_map('trim', $this->data['season'])
            : [];

        // Process the season values.
        if (in_array("0000", $seasons, true)) {
            // If one of the season codes is "0000", use zeros for each digit.
            $finalDigits = [0, 0, 0, 0];
        } else {
            // Otherwise, process each season string.
            $finalDigits = [0, 0, 0, 0];
            foreach ($seasons as $seasonStr) {
                // Ensure the season string has 4 characters by left-padding with zeros.
                $seasonStr = str_pad($seasonStr, 4, '0', STR_PAD_LEFT);
                for ($i = 0; $i < 4; $i++) {
                    // Sum the digit at each position and cap the result at 1.
                    $finalDigits[$i] = min($finalDigits[$i] + intval($seasonStr[$i]), 1);
                }
            }
        }
        // Join the digits to form the season string.
        $season = implode('', $finalDigits);

        // Combine all parts of the file name with underscores and append the .png extension.
        $fileNameParts = [
            $productType,
            $imageType,
            $imageSource,
            $countryCode,
            $productKind,
            $productNumber,
            $market,
            $focusGroup,
            $style,
            $productionType,
            $material,
            $season,
            $version,
            $brand
        ];

        $fileName = implode('_', $fileNameParts) . ('description' === $imageType)?'.txt':'.png';

        // Return the generated file name.
        return $fileName;
    }
}