<?php
require_once '../../../config.php';

ini_set('display_errors', 0);
error_reporting(0);

ob_start();

// Ensure clean JSON response
header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');

$data = $_POST;

// More verbose CSRF token checking
if (!isset($_SESSION['csrfToken'])) {
    ob_end_clean();
    echo json_encode([
        "success" => false,
        "message" => "CSRF token not set in session",
        "debug" => [
            "session_csrf" => $_SESSION['csrfToken'] ?? 'NOT SET',
            "post_csrf" => $_POST['csrfToken'] ?? 'NOT SET'
        ]
    ]);
    exit;
}

// Verify CSRF token with more robust checking
if (!isset($_SESSION['csrfToken']) || !isset($_POST['csrfToken']) || $_POST['csrfToken'] !== $_SESSION['csrfToken']) {
    // Clear output buffer
    ob_end_clean();
    echo json_encode([
        "success" => false,
        "message" => "CSRF token invalid"
    ]);
    exit;
}

// Comprehensive input validation
if (!isset($data['username'], $data['password'], $data['email'])) {
    // Clear output buffer
    ob_end_clean();
    echo json_encode([
        "success" => false,
        "message" => "Missing required fields"
    ]);
    exit;
}

// Validate input values
if (empty($data['username']) || empty($data['password']) || empty($data['email'])) {
    // Clear output buffer
    ob_end_clean();
    echo json_encode([
        "success" => false,
        "message" => "Empty required fields"
    ]);
    exit;
}

try {
    // Sanitize and validate inputs
    $username = trim($data['username']);
    $password = $data['password'];
    $email = filter_var(trim($data['email']), FILTER_VALIDATE_EMAIL);

    // Validate email
    if (!$email) {
        throw new Exception("Invalid email format");
    }

    // Validate username (e.g., alphanumeric, min/max length)
    if (!preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
        throw new Exception("Invalid username format");
    }

    // Validate password strength
    if (strlen($password) < 8) {
        throw new Exception("Password must be at least 8 characters long");
    }

    $db = PROJECT_DIR.'/data/users.db';
    $userManager = new UserManager($db);

    // Attempt to create user
    $user_id = $userManager->createUser([
        'username' => $username,
        'password' => $password,
        'email' => $email
    ]);

    // Clear output buffer
    ob_end_clean();

    if ($user_id) {
        echo json_encode([
            "success" => true,
            "message" => "User created with ID: $user_id",
            "user_id" => $user_id
        ]);
    } else {
        echo json_encode([
            "success" => false,
            "message" => "User creation failed. Username or email might already exist."
        ]);
    }
} catch (Exception $e) {
    // Clear output buffer
    ob_end_clean();

    // Log error securely
    error_log("User creation error: " . $e->getMessage());
    echo json_encode([
        "success" => false,
        "message" => $e->getMessage() // More specific error message
    ]);
}
exit;